    // js/scanner.js

    // --- DOM References ---
    const scannerModal = document.getElementById('scannerModal');
    const scannerVideo = document.getElementById('scannerVideo');
    const closeScannerBtn = document.getElementById('closeScannerBtn');
    const scannerMessage = document.getElementById('scannerMessage');

    // --- Scanner State ---
    let codeReader = null; // ZXing রিডার ইনস্ট্যান্স
    let currentTargetInputId = null; // যে ইনপুট ফিল্ড আপডেট করা হবে তার ID
    let isScannerActive = false; // স্ক্যানার সক্রিয় আছে কিনা তার ফ্ল্যাগ
    let currentStream = null; // Media stream স্টোর করার জন্য

    // --- Scanner Functions ---
    /**
     * বারকোড স্ক্যানার শুরু করুন।
     * @param {string} targetInputId - স্ক্যান ফলাফল পাওয়ার পর যে ইনপুট আপডেট হবে তার DOM ID।
     */
    async function startScanner(targetInputId) {
        if (isScannerActive || !targetInputId) {
            console.log("Scanner busy or no target.");
            return;
        }
        console.log("Starting scanner for:", targetInputId);

        if (typeof ZXing === 'undefined' || typeof ZXing.BrowserMultiFormatReader === 'undefined') {
            console.error("ZXing library not loaded!");
            scannerMessage.textContent = "Error: Scanner library failed to load.";
            scannerMessage.className = 'text-center mt-3 text-sm text-red-600';
            setTimeout(stopScanner, 3000); // 3 সেকেন্ড পর বন্ধ করুন
            return;
        }

        try {
            // আগের কোনো রিডার ইনস্ট্যান্স থাকলে ডিসপোজ করুন
            if (codeReader) {
                codeReader.reset();
                codeReader = null;
                 console.log("Disposed of previous ZXing instance.");
            }
            codeReader = new ZXing.BrowserMultiFormatReader();

            isScannerActive = true;
            currentTargetInputId = targetInputId; // টার্গেট সেট করুন
            scannerMessage.textContent = "Requesting camera access...";
            scannerMessage.className = 'text-center mt-3 text-sm text-gray-600'; // মেসেজ রিসেট করুন
            scannerModal.style.display = 'block';

            // ডিফল্ট ক্যামেরা থেকে ডিকোডিং শুরু করুন
            codeReader.decodeFromVideoDevice(undefined, 'scannerVideo', (result, err, controls) => {
                // স্ক্যানার বন্ধ করা হয়েছে কিনা চেক করুন
                if (!isScannerActive) {
                     console.log("Scan callback ignored, scanner stopped.");
                     // controls এখানে সবসময় উপলব্ধ নাও থাকতে পারে বা ঠিকভাবে কাজ নাও করতে পারে।
                     // stopScanner() ফাংশন codeReader.reset() কল করে যা stream বন্ধ করে।
                     return;
                }

                if (result) {
                    console.log("Barcode found:", result.getText());
                    scannerMessage.textContent = `Scanned: ${escapeHTML(result.getText())}`; // ফলাফল escape করুন
                    scannerMessage.className = 'text-center mt-3 text-sm text-green-600'; // সফল হলে সবুজ
                    // ঐচ্ছিকভাবে একটি শব্দ বাজান
                    // const audio = new Audio('scan-success.mp3'); // অডিও ফাইলের প্রয়োজন হবে
                    // audio.play().catch(e => console.error("Audio play failed:", e)); // প্রমিজ হ্যান্ডলিং

                    handleScanResult(result.getText()); // ফলাফল হ্যান্ডেল করুন, যা stopScanner() কল করবে

                } else if (err) {
                    // শুধুমাত্র গুরুত্বপূর্ণ ত্রুটি লগ করুন, NotFoundException উপেক্ষা করুন
                    if (!(err instanceof ZXing.NotFoundException)) {
                         console.error("Scanning Error:", err);
                         // স্ক্যানিং চলাকালীন মেসেজ এরিয়াকে error দিয়ে ভর্তি করা এড়ান
                         // scannerMessage.textContent = `Scan Error: ${err.name}`;
                         // scannerMessage.className = 'text-center mt-3 text-sm text-red-600';
                    } else {
                         // যদি আগে error/success দেখানো হয়ে থাকে, মেসেজ রিসেট করুন
                         if (scannerMessage.textContent !== "Point camera at barcode...") {
                             scannerMessage.textContent = "Point camera at barcode...";
                             scannerMessage.className = 'text-center mt-3 text-sm text-gray-600';
                         }
                    }
                }
            })
            .then(ctl => {
                 console.log("DecodeFromVideoDevice resolved. Scanner active.");
                 // Media stream স্টোর করুন যাতে পরে এটি বন্ধ করা যায়
                 if (scannerVideo.srcObject) {
                     currentStream = scannerVideo.srcObject;
                 }
                 scannerMessage.textContent = "Point camera at barcode..."; // ক্যামেরা চালু হলে এই মেসেজ দেখান
            })
            .catch(err => {
                console.error("Error in decodeFromVideoDevice setup:", err);
                isScannerActive = false;
                // সুনির্দিষ্ট ত্রুটি হ্যান্ডেল করুন
                let errorMessage = 'Error accessing camera.';
                if (err.name === 'NotAllowedError') {
                    errorMessage = "Camera access denied. Please allow in browser settings.";
                } else if (err.name === 'NotFoundError' || err.name === 'DevicesNotFoundError') {
                     errorMessage = "No camera found.";
                } else if (err.name === 'NotReadableError') {
                    errorMessage = "Camera is already in use or unreadable.";
                } else {
                    errorMessage = `Camera error: ${err.name}`;
                }
                scannerMessage.textContent = errorMessage;
                scannerMessage.className = 'text-center mt-3 text-sm text-red-600';
                // যদি কোনো স্থায়ী ত্রুটি হয়, বন্ধ করুন এবং অবিলম্বে স্বয়ংক্রিয়ভাবে বন্ধ করবেন না
                 setTimeout(stopScanner, 4000); // 4 সেকেন্ড পর বন্ধ করুন যাতে ব্যবহারকারী মেসেজ দেখতে পায়
            });

        } catch (error) {
            console.error("Error initializing ZXing:", error);
            scannerMessage.textContent = `Initialization Error: ${error.message}`;
            scannerMessage.className = 'text-center mt-3 text-sm text-red-600';
            isScannerActive = false;
            setTimeout(stopScanner, 4000); // ইনিট ত্রুটি দেখানোর পর স্বয়ংক্রিয়ভাবে বন্ধ করুন
        }
    }

    /**
     * স্ক্যানার বন্ধ করুন এবং ক্যামেরা রিলিজ করুন।
     */
    function stopScanner() {
        console.log("Stopping scanner...");
        if (!isScannerActive && !codeReader && !currentStream) {
             console.log("Scanner already stopped or not active.");
             scannerModal.style.display = 'none'; // নিশ্চিত করুন মডেল বন্ধ আছে
             return;
        }

        isScannerActive = false; // ফ্ল্যাগ সেট করুন অবিলম্বে

         if (codeReader) {
             try {
                codeReader.reset(); // ডিকোডিং স্ট্রিম বন্ধ করে এবং ক্যামেরা রিলিজ করে
                 console.log("ZXing Reader reset called.");
             } catch (e) {
                 console.error("Error resetting codeReader:", e);
             }
             codeReader = null; // ইনস্ট্যান্স ক্লিয়ার করুন
         }

         // Media stream ট্র্যাকগুলি স্পষ্টভাবে বন্ধ করুন, যদি reset() সম্পূর্ণভাবে কাজ না করে
         if (currentStream) {
             console.log("Stopping media stream tracks.");
             currentStream.getTracks().forEach(track => {
                  try {
                      track.stop();
                  } catch(e) {
                      console.error("Error stopping track:", e);
                  }
             });
             currentStream = null;
         }
         // Fallback যদি srcObject সেট করা থাকে (ZXing এর সাথে কম সাধারণ)
         if (scannerVideo.srcObject) {
              try {
                 scannerVideo.srcObject.getTracks().forEach(track => track.stop());
              } catch(e) {
                  console.error("Error stopping srcObject tracks:", e);
              }
             scannerVideo.srcObject = null;
         }


        // ভিডিও এলিমেন্টের স্টেট ক্লিয়ার করুন
        scannerVideo.removeAttribute('src');
        // scannerVideo.load(); // load() কল করার প্রয়োজন নেই যদি srcAttribute remove করা হয়

        scannerModal.style.display = 'none';
        scannerMessage.textContent = "Requesting camera..."; // পরবর্তী ব্যবহারের জন্য মেসেজ রিসেট করুন
        scannerMessage.className = 'text-center mt-3 text-sm text-gray-600';
        currentTargetInputId = null; // টার্গেট ক্লিয়ার করুন
         console.log("Scanner stopped successfully.");
    }

    /**
     * স্ক্যান ফলাফলের হ্যান্ডলার।
     * @param {string} scannedData - স্ক্যান করা ডেটা (যেমন বারকোড)।
     */
    function handleScanResult(scannedData) {
         if (!currentTargetInputId) {
            console.warn("Scan result received but no target input ID set.");
            stopScanner(); // টার্গেট না থাকলে বন্ধ করে দিন
            return;
        }

        const targetInput = document.getElementById(currentTargetInputId);
        if (targetInput) {
            console.log(`Updating input ${currentTargetInputId} with data: ${scannedData}`);
            const trimmedData = scannedData.trim();
            targetInput.value = trimmedData;

            // যেকোনো লিঙ্কড লজিক (যেমন ডুপ্লিকেট চেক) ট্রিগার করার জন্য ইভেন্ট ডিসপ্যাচ করুন
            try {
                console.log('Dispatching input event for', currentTargetInputId);
                // একটি 'input' ইভেন্ট তৈরি করুন এবং ডিসপ্যাচ করুন
                const inputEvent = new Event('input', { bubbles: true, cancelable: true });
                targetInput.dispatchEvent(inputEvent);

                 // একটি ছোট বিলম্বের পর 'blur' ইভেন্ট ডিসপ্যাচ করুন ফোকাস হারানোর সিমুলেশন করার জন্য
                 // এটি ডুপ্লিকেট চেক বা অন্যান্য validation ট্রিগার করতে পারে
                 setTimeout(() => {
                      console.log('Dispatching blur event for', currentTargetInputId);
                      const blurEvent = new Event('blur', { bubbles: true, cancelable: true });
                       targetInput.dispatchEvent(blurEvent);
                 }, 50); // ছোট বিলম্ব

            } catch(e) {
                 console.error("Error dispatching events:", e);
            }

            // ঐচ্ছিকভাবে ইনপুট ফিল্ড ফোকাস করুন
            // targetInput.focus(); // workflow এর উপর নির্ভর করে
        } else {
             console.error(`Target input field with ID ${currentTargetInputId} not found!`);
        }
        stopScanner(); // ফলাফল প্রক্রিয়াকরণের পর স্ক্যানার বন্ধ করুন
    }

    /**
     * বারকোড স্ক্যান বাটনে ক্লিক হ্যান্ডেল করুন।
     * @param {Event} event - ক্লিক ইভেন্ট।
     */
    function handleScanButtonClick(event) {
        const button = event.target.closest('.scan-barcode-btn');
        if (!button) return;
        console.log('Scan button clicked:', button);

        const targetId = button.dataset.target;
        if (!targetId) {
            console.error('Scan button is missing data-target attribute.');
            alert('Scanner button is not configured correctly.');
            return;
        }

        const targetInput = document.getElementById(targetId);
         if (!targetInput) {
             console.error('Target input field not found for ID:', targetId);
             alert(`Error: Input field with ID '${targetId}' not found.`);
             return;
         }

         // অন্য কোনো স্ক্যান সক্রিয় থাকলে, প্রথমে সেটি বন্ধ করুন
         if (isScannerActive) {
             console.warn("Stopping existing scanner before starting new one.");
             stopScanner();
             // আগের স্টপ প্রক্রিয়া শেষ হওয়ার জন্য একটু সময় দিন
             setTimeout(() => {
                startScanner(targetId); // টার্গেট সেট করে স্ক্যানার শুরু করুন
             }, 100); // সামান্য বিলম্ব
         } else {
             startScanner(targetId); // টার্গেট সেট করে স্ক্যানার শুরু করুন
         }
    }

    // মডাল বন্ধ করার হ্যান্ডলার সেটআপ করুন (scanner.js এর মধ্যে)
    function setupScannerModalHandlers() {
        if(closeScannerBtn) closeScannerBtn.addEventListener('click', stopScanner);
        if(scannerModal) {
            scannerModal.addEventListener('click', (event) => {
                if (event.target === scannerModal) stopScanner();
            });
             // ঐচ্ছিকভাবে ভিডিও কন্টেইনারের ভিতরে ক্লিক করলে স্ক্যানার বন্ধ হওয়া থেকে বিরত রাখুন
             scannerModal.querySelector('.modal-content > .relative')?.addEventListener('click', (e) => {
                 e.stopPropagation();
             });
        }
    }

    // DOMContentLoaded এর পর কল করার জন্য ফাংশন এক্সপোর্ট করুন
    // script.js এই ফাংশন কল করবে
    export { setupScannerModalHandlers, handleScanButtonClick, startScanner, stopScanner };

