// js/api.js

// API এন্ডপয়েন্ট URL
const API_URL = 'backend.php'; // আপনার backend.php ফাইলের পাথ

/**
 * API কল করার জেনেরিক ফাংশন।
 * ত্রুটি হ্যান্ডলিং এবং JSON পার্সিং অন্তর্ভুক্ত।
 * @param {string} action - ব্যাকএন্ডে পাঠানো অ্যাকশন।
 * @param {FormData|Object} data - ফর্ম ডেটা বা অবজেক্ট।
 * @param {string} method - HTTP মেথড (সাধারণত 'POST')।
 * @returns {Promise<Object>} JSON রেসপন্স ডেটা সহ প্রমিজ।
 * @throws {Error} নেটওয়ার্ক বা API ত্রুটির জন্য।
 */
async function callApi(action, data = {}, method = 'POST') {
     const formData = new FormData();
     formData.append('action', action);

     if (data instanceof FormData) {
         // যদি ডেটা ইতিমধ্যেই FormData হয়, বিদ্যমান এন্ট্রিকপি করুন
         for (let pair of data.entries()) {
             // যদি action ইতিমধ্যেই যোগ করা হয়ে থাকে, skip করুন
             if (pair[0] === 'action') continue;
             formData.append(pair[0], pair[1]);
         }
     } else if (typeof data === 'object' && data !== null) {
         // যদি ডেটা একটি সাধারণ অবজেক্ট হয়, FormData এ যোগ করুন
         for (const key in data) {
             if (Object.hasOwnProperty.call(data, key)) {
                 formData.append(key, data[key]);
             }
         }
     }

    try {
        const response = await fetch(API_URL, {
            method: method,
            // FormData ব্যবহার করলে Content-Type হেডার সেট করার প্রয়োজন নেই, ব্রাউজার নিজেই boundary সহ সেট করে
            body: formData
        });

        // নিশ্চিত করুন রেসপন্স JSON ফর্ম্যাটে আছে
         const contentType = response.headers.get("content-type");
         if (!contentType || !contentType.includes("application/json")) {
             const textResponse = await response.text();
              console.error("Non-JSON response from server:", { status: response.status, statusText: response.statusText, body: textResponse.substring(0, 500) });
             // লগইন চেক করুন, যদি 401 Unauthorized হয়, সম্ভবত সেশন এক্সপায়ার হয়েছে
             if (response.status === 401 || textResponse.includes('Authentication required')) {
                  // ui.showLogin() কল করা script.js এর দায়িত্ব
                  // এখানে Error throw না করে, নির্দিষ্ট API ফাংশনকে auth error হ্যান্ডেল করতে দিন
                  // throw new Error('Authentication required. Please log in again.');
             }
             throw new Error(`Server returned non-JSON response: ${response.status} ${response.statusText}`);
         }

        const result = await response.json();

        // API ত্রুটি হ্যান্ডেল করুন (backend.php success: false রিটার্ন করলে)
        if (!result.success) {
             // API call backend পর্যন্ত সফলভাবে পৌঁছেছে এবং structurad response রিটার্ন করেছে,
             // কিন্তু operation নিজেই ব্যর্থ হয়েছে (যেমন, invalid login, auth required, validation error)।
             // backend থেকে আসা মেসেজ সহ একটি Error object throw করুন।
             const error = new Error(result.message || 'API request failed.');
             error.apiSuccess = false; // এটি backend থেকে আসা failed response নির্দেশ করে
             error.apiMessage = result.message; // original message স্টোর করুন
             throw error;
        }

        return result; // success: true সহ JSON ডেটা রিটার্ন করুন

    } catch (error) {
        console.error("API call error:", error);
        // এই catch block টি network errors বা non-JSON response এর জন্য।
        // Backend থেকে আসা errors (success: false) উপরের block এ throw করা হয়।
        // Error টি পুনরায় throw করুন যাতে calling function এটি catch করতে পারে।
        throw error;
    }
}

// --- API specific functions ---

export async function apiLogin(username, password) {
    const data = { username: username, password: password };
    return callApi('login', data); // callApi কে login এর success/failure হ্যান্ডেল করতে দিন
}

export async function apiLogout() {
    return callApi('logout'); // callApi কে logout এর success/failure হ্যান্ডেল করতে দিন
}

export async function apiCheckAuth() {
     try {
         const result = await callApi('check_auth');
         // If callApi returns result, it means success: true from backend
         return true;
     } catch (error) {
          // If callApi throws, it means success was false or a network error occurred.
          // Check if it's an expected "Not authenticated" error from backend (apiSuccess === false).
         if (error.apiSuccess === false && error.apiMessage && error.apiMessage.includes('Authentication required')) {
              // এটি প্রত্যাশিত scenario যখন logged in না থাকে। শুধু false রিটার্ন করুন।
              console.log("Auth check API result: Not authenticated (expected)."); // Log as info, not error
             return false;
         } else {
              // এটি একটি অপ্রত্যাশিত error (network, server issue, etc.)
              console.error("Auth check API error:", error.message); // Log as error
              // Desired behavior এর উপর নির্ভর করে, re-throw করতে পারেন বা false রিটার্ন করতে পারেন
             throw error; // অপ্রত্যাশিত error গুলো re-throw করুন
             // return false; // অথবা যেকোনো failure এর জন্য false রিটার্ন করুন
         }
     }
}


export async function apiFetchRequests() {
     try {
         const result = await callApi('fetch_requests');
         return result.requests || [];
     } catch (error) {
         // এই catch block টি auth errors (callApi দ্বারা thrown হওয়া) এবং অন্যান্য errors হ্যান্ডেল করবে
         // Authentication required errors thrown by callApi are caught here.
         // script.js এ api calls এর জন্য একটি top-level catch block থাকা উচিত authentication error পেলে login এ redirect করার জন্য।
         throw error;
     }
}

export async function apiFetchRequestById(id) {
    const data = { id: id };
     try {
        const result = await callApi('fetch_request_by_id', data);
         return result.request;
     } catch (error) {
         throw error;
     }
}

export async function apiApproveRequest(formData) {
    // formData তে action='approve_request' যোগ করার দরকার নেই, callApi function যোগ করে দেবে
     try {
        const result = await callApi('approve_request', formData);
         return result; // { success: bool, message: string } রিটার্ন করুন
     } catch (error) {
         throw error;
     }
}

export async function apiAddProduct(formData) {
     // formData তে action='add_product' যোগ করার দরকার নেই
     try {
        const result = await callApi('add_product', formData);
         return result; // { success: bool, message: string } রিটার্ন করুন
     } catch (error) {
         throw error;
     }
}

export async function apiDeleteRequest(id) {
     const data = { id: id };
     try {
        const result = await callApi('delete_request', data);
         return result; // { success: bool, message: string } রিটার্ন করুন
     } catch (error) {
         throw error;
     }
}

export async function apiCheckDuplicate(productName, barcode) {
     const data = { ProductName: productName, Barcode: barcode };
     try {
         const result = await callApi('check_duplicate', data);
         // Duplicate check এর জন্য, success: false মানে "error" নয়, বরং duplicate আছে।
         // আমরা এখানে callApi থেকে আসা error re-throw করব না যদি apiSuccess false হয়।
         return {
             success: true, // এটি বোঝায় যে check প্রক্রিয়াটি সফল হয়েছে
             isDuplicateName: result.isDuplicateName,
             isDuplicateBarcode: result.isDuplicateBarcode,
             message: result.message // backend message থাকলে যোগ করুন
         };
     } catch (error) {
         // এই catch টি network errors বা check চলাকালীন unexpected backend issues এর জন্য।
         console.error("Duplicate check network or API error:", error.message);
          // check করতে failure হলে, safe default হিসেবে কোনো duplicate পাওয়া যায়নি এমন রিটার্ন করুন
         return { success: false, isDuplicateName: false, isDuplicateBarcode: false, message: error.message };
     }
}

// প্রয়োজন অনুযায়ী অন্যান্য API কলিং ফাংশন এখানে যোগ করা যেতে পারে
// export { callApi }; // Optionally export callApi if needed elsewhere, but not strictly necessary for this structure.